#ifndef BINARY_TREE_H
#define BINARY_TREE_H

#include"BinaryNode.h"
#include "Queue.h"
#include "Stack.h"

template<class Type>
class BinaryTree
{
protected:
	BinaryNode<Type>* rootPtr; // Pointer to the root node

	int count; // Number of Nodes in the Tree

public:

	//Constructors and Destructors and Admin Functions
	BinaryTree() { rootPtr = 0; count = 0; }
	BinaryTree(const BinaryTree<Type>&);
	virtual ~BinaryTree() { destroyTree(rootPtr); count = 0; }

	//Common Functions for all Binary Trees
	bool isEmpty() const { return count == 0; }
	int size() const { return count; }
	void clear() { destroyTree(rootPtr); rootPtr = 0; count = 0; }


	/**~*~*
	Recursive Traversal Methods
	*~**/

	//In Order
	template <class X>
	void inOrder(void visit(const X&, ostream&), ostream& os) const
	{
		_inOrder(visit, os, rootPtr);
	}

	//Pre Order
	template <class X>
	void preOrder(void visit(const X&, ostream&), ostream& os) const
	{
		_preOrder(visit, os, rootPtr);
	}

	//Post Order
	template <class X>
	void postOrder(void visit(const X&, ostream&), ostream& os) const
	{
		_postOrder(visit, os, rootPtr);
	}

	/**~*~*
	Breadthfirst traversal Methods
	*~**/
	template <class X>
	void breadthFirst(void visit(const X&, ostream&), ostream& os) const
	{
		Queue<Type> queue;
		_breadthFirst(visit, os, rootPtr, queue);
	}


	//Abstract functions that the Binary Search Tree need to define
	virtual bool insert(const Type&) = 0;
	virtual bool remove(const Type&) = 0;
	virtual bool getEntry(const Type&, Type&) const = 0;

private:

	// delete all nodes from the tree
	void destroyTree(BinaryNode<Type>*);

	//Internal Traversal functions

	//Breadth First
	template <class X>
	void _breadthFirst(void visit(const X&, ostream&), ostream&, BinaryNode<Type>*, Queue<Type>&) const;

	//InOrder
	template<class X>
	void _inOrder(void visit(const X&, ostream&), ostream& os, BinaryNode<Type>*) const;

	//PreOrder
	template<class X>
	void _preOrder(void visit(const X&, ostream&), ostream& os, BinaryNode<Type>*) const;

	//PostOrder
	template<class X>
	void _postOrder(void visit(const X&, ostream&), ostream& os, BinaryNode<Type>*) const;
};

/**~*~*
Destroys tree recursively.
*~**/

template<class Type>
void BinaryTree<Type>::destroyTree(BinaryNode<Type>* nodePtr)
{
	count = 0;

	if (!nodePtr)
		return;

	destroyTree(nodePtr->getLeftPtr());
	destroyTree(nodePtr->getRightPtr());
	delete nodePtr;
}

/**~*~*
For Breadth First traversal.
*~**/
template <class Type> template<class X>
void BinaryTree<Type>::_breadthFirst(void visit(const X&, ostream&), ostream& os, BinaryNode<Type>* nodePtr, Queue<Type>& queue) const
{
	if (!nodePtr)
		return;
	queue.enQueue(nodePtr->getItem());
	_breadthFirst(visit, os, nodePtr->getRightPtr(), queue);
	_breadthFirst(visit, os, nodePtr->getLeftPtr(), queue);
	Type curr;
	queue.deQueue(curr);
	visit(curr);
}

/**~*~*
For recursive inorder traversal.
*~**/
template<class Type> template<class X>
void BinaryTree<Type>::_inOrder(void visit(const X&, ostream&), ostream& os, BinaryNode<Type>* nodePtr) const
{
	if (!nodePtr)
		return;

	_inOrder(visit, os, nodePtr->getLeftPtr());
	visit(nodePtr->getItem(), os);
	_inOrder(visit, os, nodePtr->getRightPtr());

}

/**~*~*
For recursive preorder traversal.
*~**/
template<class Type> template<class X>
void BinaryTree<Type>::_preOrder(void visit(const X&, ostream&), ostream& os, BinaryNode<Type>* nodePtr) const
{
	if (!nodePtr)
		return;

	
	visit(nodePtr->getItem(), os);
	_preOrder(visit, os, nodePtr->getLeftPtr());
	_preOrder(visit, os, nodePtr->getRightPtr());
}

/**~*~*
For recursive postorder traversal.
*~**/
template<class Type> template<class X>
void BinaryTree<Type>::_postOrder(void visit(const X&, ostream&), ostream& os, BinaryNode<Type>* nodePtr) const
{
	if (!nodePtr)
		return;


	_postOrder(visit, os, nodePtr->getLeftPtr());
	_postOrder(visit, os, nodePtr->getRightPtr());
	visit(nodePtr->getItem(), os);
}

#endif